/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.servicecomb.transport.rest.client.http;

import java.util.Collection;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;

import jakarta.ws.rs.core.HttpHeaders;
import jakarta.ws.rs.core.MediaType;

import org.apache.servicecomb.common.rest.RestConst;
import org.apache.servicecomb.common.rest.codec.produce.ProduceEventStreamProcessor;
import org.apache.servicecomb.common.rest.codec.produce.ProduceProcessor;
import org.apache.servicecomb.common.rest.codec.produce.ProduceProcessorManager;
import org.apache.servicecomb.common.rest.definition.RestOperationMeta;
import org.apache.servicecomb.common.rest.filter.HttpClientFilter;
import org.apache.servicecomb.core.Invocation;
import org.apache.servicecomb.core.definition.OperationMeta;
import org.apache.servicecomb.foundation.vertx.http.HttpServletRequestEx;
import org.apache.servicecomb.foundation.vertx.http.HttpServletResponseEx;
import org.apache.servicecomb.swagger.invocation.Response;
import org.apache.servicecomb.swagger.invocation.context.HttpStatus;
import org.apache.servicecomb.swagger.invocation.exception.CommonExceptionData;
import org.apache.servicecomb.swagger.invocation.exception.InvocationException;
import org.apache.servicecomb.swagger.invocation.sse.SseEventResponseEntity;
import org.reactivestreams.Publisher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.databind.JavaType;
import com.netflix.config.DynamicPropertyFactory;

import io.reactivex.rxjava3.core.Flowable;
import io.vertx.core.buffer.Buffer;

public class DefaultHttpClientFilter implements HttpClientFilter {
  private static final Logger LOGGER = LoggerFactory.getLogger(DefaultHttpClientFilter.class);

  private static final boolean enabled = DynamicPropertyFactory.getInstance().getBooleanProperty
      ("servicecomb.http.filter.client.default.enabled", true).get();

  private static final String PRINT_ERROR_RESPONSE = "servicecomb.http.filter.client.default.shouldPrintErrorResponse";

  @Override
  public int getOrder() {
    return 10000;
  }

  @Override
  public CompletableFuture<Void> beforeSendRequestAsync(Invocation invocation, HttpServletRequestEx requestEx) {
    return CompletableFuture.completedFuture(null);
  }

  @Override
  public boolean enabled() {
    return enabled;
  }

  protected ProduceProcessor findProduceProcessor(RestOperationMeta restOperation,
      HttpServletResponseEx responseEx) {
    String contentType = responseEx.getHeader(HttpHeaders.CONTENT_TYPE);
    if (contentType == null) {
      return null;
    }

    String contentTypeForFind = contentType;
    int idx = contentType.indexOf(';');
    if (idx != -1) {
      contentTypeForFind = contentType.substring(0, idx);
    }

    return restOperation.findProduceProcessor(contentTypeForFind);
  }

  protected Response extractResponse(Invocation invocation, HttpServletResponseEx responseEx) {
    Object result = invocation.getHandlerContext().get(RestConst.READ_STREAM_PART);
    if (result != null) {
      return Response.create(responseEx.getStatusType(), result);
    }
    OperationMeta operationMeta = invocation.getOperationMeta();
    JavaType responseType = invocation.findResponseType(responseEx.getStatus());
    RestOperationMeta swaggerRestOperation = operationMeta.getExtData(RestConst.SWAGGER_REST_OPERATION);
    ProduceProcessor produceProcessor = findProduceProcessor(swaggerRestOperation, responseEx);
    if (produceProcessor == null && !isEventStream(responseEx)) {
      // This happens outside the runtime such as Servlet filter response. Here we give a default json parser to it
      // and keep user data not get lose.
      LOGGER.warn("Response content-type {} is not supported. Method {}, path {}, statusCode {}, reasonPhrase {}.",
          responseEx.getHeader(HttpHeaders.CONTENT_TYPE),
          swaggerRestOperation.getHttpMethod(),
          swaggerRestOperation.getAbsolutePath(),
          responseEx.getStatus(),
          responseEx.getStatusType().getReasonPhrase()
      );
      produceProcessor = ProduceProcessorManager.INSTANCE.findDefaultProcessor();
    }

    try {
      if (responseEx.getFlowableBuffer() == null) {
        result = produceProcessor.decodeResponse(responseEx.getBodyBuffer(), responseType);
      } else {
        Flowable<Buffer> flowable = responseEx.getFlowableBuffer();
        ProduceEventStreamProcessor finalProduceProcessor = new ProduceEventStreamProcessor();
        result = flowable.concatMap(buffer -> extractFlowableBody(finalProduceProcessor, responseType, buffer))
            .doFinally(finalProduceProcessor::close)
            .doOnCancel(finalProduceProcessor::close)
            .filter(Objects::nonNull);
      }
      Response response = Response.create(responseEx.getStatusType(), result);
      if (response.isFailed()) {
        LOGGER.warn("invoke operation [{}] failed, status={}, msg={}", invocation.getMicroserviceQualifiedName(),
            responseEx.getStatusType().getStatusCode(), result == null ? "" : result.toString());
      }
      return response;
    } catch (Exception e) {
      if (DynamicPropertyFactory.getInstance().getBooleanProperty(PRINT_ERROR_RESPONSE, false).get()) {
        LOGGER.error("failed to decode response body. response is {}", responseEx.getBodyBuffer());
      } else {
        LOGGER.error("failed to decode response body.", e);
      }
      String msg =
          String.format("Failed to decode response body. Operation %s.",
              invocation.getMicroserviceQualifiedName());
      if (HttpStatus.isSuccess(responseEx.getStatus())) {
        return Response.createConsumerFail(
            new InvocationException(400, responseEx.getStatusType().getReasonPhrase(),
                new CommonExceptionData(msg), e));
      }
      return Response.createConsumerFail(
          new InvocationException(responseEx.getStatus(), responseEx.getStatusType().getReasonPhrase(),
              new CommonExceptionData(msg), e));
    }
  }

  private boolean isEventStream(HttpServletResponseEx responseEx) {
    return responseEx.getHeader(HttpHeaders.CONTENT_TYPE) != null
        && responseEx.getHeader(HttpHeaders.CONTENT_TYPE).contains(MediaType.SERVER_SENT_EVENTS);
  }

  protected Publisher<SseEventResponseEntity<?>> extractFlowableBody(ProduceEventStreamProcessor produceProcessor,
      JavaType responseType, Buffer buffer) throws Exception {
    return produceProcessor.decodeResponse(buffer, responseType);
  }

  @Override
  public Response afterReceiveResponse(Invocation invocation, HttpServletResponseEx responseEx) {
    Response response = extractResponse(invocation, responseEx);

    for (String headerName : responseEx.getHeaderNames()) {
      if (headerName.equals(":status")) {
        continue;
      }
      Collection<String> headerValues = responseEx.getHeaders(headerName);
      for (String headerValue : headerValues) {
        response.addHeader(headerName, headerValue);
      }
    }

    return response;
  }
}
