/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.lucene.analysis.hunspell;

import java.util.Collections;
import java.util.List;

/**
 * Suggestion to add/edit dictionary entries to generate a given list of words created by {@link
 * WordFormGenerator#compress}.
 */
public class EntrySuggestion {
  private final List<DictEntry> toEdit, toAdd;
  private final List<String> extraGenerated;

  EntrySuggestion(List<DictEntry> toEdit, List<DictEntry> toAdd, List<String> extraGenerated) {
    this.toEdit = Collections.unmodifiableList(toEdit);
    this.toAdd = Collections.unmodifiableList(toAdd);
    this.extraGenerated = Collections.unmodifiableList(extraGenerated);
  }

  /**
   * @return the existing dictionary entries whose flags would need changing to accommodate the
   *     given word list
   */
  public List<DictEntry> getEntriesToEdit() {
    return toEdit;
  }

  /**
   * @return new dictionary entries to be added to accommodate the given word list
   */
  public List<DictEntry> getEntriesToAdd() {
    return toAdd;
  }

  /**
   * @return additional words generated by union of {@link #getEntriesToAdd()} and {@link
   *     #getEntriesToEdit()} which weren't in the given list of words
   */
  public List<String> getExtraGeneratedWords() {
    return extraGenerated;
  }

  @Override
  public String toString() {
    return "EntrySuggestion{" + internalsToString() + '}';
  }

  String internalsToString() {
    return "toEdit=" + toEdit + ", toAdd=" + toAdd + ", extra=" + extraGenerated;
  }
}
