/*
 * Decompiled with CFR 0.152.
 */
package org.apache.lucene.codecs.bloom;

import java.io.IOException;
import org.apache.lucene.codecs.bloom.HashFunction;
import org.apache.lucene.codecs.bloom.MurmurHash64;
import org.apache.lucene.store.DataInput;
import org.apache.lucene.store.DataOutput;
import org.apache.lucene.util.Accountable;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.FixedBitSet;
import org.apache.lucene.util.RamUsageEstimator;

public class FuzzySet
implements Accountable {
    private HashFunction hashFunction;
    private FixedBitSet filter;
    private int bloomSize;
    private final int hashCount;
    static final int[] usableBitSetSizes = new int[26];

    public static int getNearestSetSize(int maxNumberOfBits) {
        int result = usableBitSetSizes[0];
        for (int i = 0; i < usableBitSetSizes.length; ++i) {
            if (usableBitSetSizes[i] > maxNumberOfBits) continue;
            result = usableBitSetSizes[i];
        }
        return result;
    }

    public static int getNearestSetSize(int maxNumberOfValuesExpected, float desiredSaturation) {
        for (int i = 0; i < usableBitSetSizes.length; ++i) {
            int numSetBitsAtDesiredSaturation = (int)((float)usableBitSetSizes[i] * desiredSaturation);
            int estimatedNumUniqueValues = FuzzySet.getEstimatedNumberUniqueValuesAllowingForCollisions(usableBitSetSizes[i], numSetBitsAtDesiredSaturation);
            if (estimatedNumUniqueValues <= maxNumberOfValuesExpected) continue;
            return usableBitSetSizes[i];
        }
        return -1;
    }

    public static FuzzySet createSetBasedOnMaxMemory(int maxNumBytes) {
        int setSize = FuzzySet.getNearestSetSize(maxNumBytes);
        return new FuzzySet(new FixedBitSet(setSize + 1), setSize, 1);
    }

    public static FuzzySet createSetBasedOnQuality(int maxNumUniqueValues, float desiredMaxSaturation, int version) {
        int setSize = FuzzySet.getNearestSetSize(maxNumUniqueValues, desiredMaxSaturation);
        return new FuzzySet(new FixedBitSet(setSize + 1), setSize, 1);
    }

    public static FuzzySet createOptimalSet(int maxNumUniqueValues, float targetMaxFpp) {
        int setSize = (int)Math.ceil((double)maxNumUniqueValues * Math.log(targetMaxFpp) / Math.log(1.0 / Math.pow(2.0, Math.log(2.0))));
        setSize = FuzzySet.getNearestSetSize(2 * setSize);
        int optimalK = (int)Math.round((double)setSize / (double)maxNumUniqueValues * Math.log(2.0));
        return new FuzzySet(new FixedBitSet(setSize + 1), setSize, optimalK);
    }

    private FuzzySet(FixedBitSet filter, int bloomSize, int hashCount) {
        this.filter = filter;
        this.bloomSize = bloomSize;
        this.hashFunction = MurmurHash64.INSTANCE;
        this.hashCount = hashCount;
    }

    public ContainsResult contains(BytesRef value) {
        long hash = this.hashFunction.hash(value);
        int msb = (int)(hash >>> 32);
        int lsb = (int)hash;
        for (int i = 0; i < this.hashCount; ++i) {
            int bloomPos = lsb + i * msb;
            if (this.mayContainValue(bloomPos)) continue;
            return ContainsResult.NO;
        }
        return ContainsResult.MAYBE;
    }

    public void serialize(DataOutput out) throws IOException {
        out.writeVInt(this.hashCount);
        out.writeInt(this.bloomSize);
        long[] bits = this.filter.getBits();
        out.writeInt(bits.length);
        for (int i = 0; i < bits.length; ++i) {
            out.writeLong(bits[i]);
        }
    }

    public static FuzzySet deserialize(DataInput in) throws IOException {
        int hashCount = in.readVInt();
        int bloomSize = in.readInt();
        int numLongs = in.readInt();
        long[] longs = new long[numLongs];
        in.readLongs(longs, 0, numLongs);
        FixedBitSet bits = new FixedBitSet(longs, bloomSize + 1);
        return new FuzzySet(bits, bloomSize, hashCount);
    }

    private boolean mayContainValue(int aHash) {
        int pos = aHash & this.bloomSize;
        return this.filter.get(pos);
    }

    public void addValue(BytesRef value) throws IOException {
        long hash = this.hashFunction.hash(value);
        int msb = (int)(hash >>> 32);
        int lsb = (int)hash;
        for (int i = 0; i < this.hashCount; ++i) {
            int bloomPos = lsb + i * msb & this.bloomSize;
            this.filter.set(bloomPos);
        }
    }

    public FuzzySet downsize(float targetMaxSaturation) {
        int numBitsSet = this.filter.cardinality();
        FixedBitSet rightSizedBitSet = this.filter;
        int rightSizedBitSetSize = this.bloomSize;
        for (int i = 0; i < usableBitSetSizes.length; ++i) {
            int candidateBitsetSize = usableBitSetSizes[i];
            float candidateSaturation = (float)numBitsSet / (float)candidateBitsetSize;
            if (!(candidateSaturation <= targetMaxSaturation)) continue;
            rightSizedBitSetSize = candidateBitsetSize;
            break;
        }
        if (rightSizedBitSetSize < this.bloomSize) {
            rightSizedBitSet = new FixedBitSet(rightSizedBitSetSize + 1);
            int bitIndex = 0;
            do {
                if ((bitIndex = this.filter.nextSetBit(bitIndex)) == Integer.MAX_VALUE) continue;
                int downSizedBitIndex = bitIndex & rightSizedBitSetSize;
                rightSizedBitSet.set(downSizedBitIndex);
                ++bitIndex;
            } while (bitIndex >= 0 && bitIndex <= this.bloomSize);
        } else {
            return null;
        }
        return new FuzzySet(rightSizedBitSet, rightSizedBitSetSize, this.hashCount);
    }

    public int getEstimatedUniqueValues() {
        return FuzzySet.getEstimatedNumberUniqueValuesAllowingForCollisions(this.bloomSize, this.filter.cardinality(), this.hashCount);
    }

    public static int getEstimatedNumberUniqueValuesAllowingForCollisions(int setSize, int numRecordedBits) {
        return FuzzySet.getEstimatedNumberUniqueValuesAllowingForCollisions(setSize, numRecordedBits, 1);
    }

    public static int getEstimatedNumberUniqueValuesAllowingForCollisions(int setSize, int numRecordedBits, int hashCount) {
        double setSizeAsDouble = setSize;
        double numRecordedBitsAsDouble = numRecordedBits;
        double hashCountAsDouble = hashCount;
        double saturation = numRecordedBitsAsDouble / setSizeAsDouble;
        double logInverseSaturation = Math.log(1.0 - saturation) * -1.0;
        return (int)(setSizeAsDouble * logInverseSaturation / hashCountAsDouble);
    }

    public float getTargetMaxSaturation() {
        return 0.5f;
    }

    public float getSaturation() {
        int numBitsSet = this.filter.cardinality();
        return (float)numBitsSet / (float)this.bloomSize;
    }

    public long ramBytesUsed() {
        return RamUsageEstimator.sizeOf((long[])this.filter.getBits());
    }

    public String toString() {
        return this.getClass().getSimpleName() + "(hash=" + String.valueOf(this.hashFunction) + ", k=" + this.hashCount + ", bits=" + this.filter.cardinality() + "/" + this.filter.length() + ")";
    }

    static {
        for (int i = 0; i < usableBitSetSizes.length; ++i) {
            FuzzySet.usableBitSetSizes[i] = (1 << i + 6) - 1;
        }
    }

    public static enum ContainsResult {
        MAYBE,
        NO;

    }
}

