/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.blob;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.security.MessageDigest;
import java.util.Arrays;
import org.apache.flink.api.common.JobID;
import org.apache.flink.core.fs.FSDataInputStream;
import org.apache.flink.core.fs.FSDataOutputStream;
import org.apache.flink.core.fs.FileSystem;
import org.apache.flink.core.fs.Path;
import org.apache.flink.runtime.blob.BlobKey;
import org.apache.flink.runtime.blob.BlobStoreService;
import org.apache.flink.runtime.blob.BlobUtils;
import org.apache.flink.shaded.guava31.com.google.common.io.Files;
import org.apache.flink.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileSystemBlobStore
implements BlobStoreService {
    private static final Logger LOG = LoggerFactory.getLogger(FileSystemBlobStore.class);
    private final FileSystem fileSystem;
    private volatile boolean basePathCreated;
    private final String basePath;
    public static final String BLOB_PATH_NAME = "blob";

    public FileSystemBlobStore(FileSystem fileSystem, String storagePath) throws IOException {
        this.fileSystem = (FileSystem)Preconditions.checkNotNull((Object)fileSystem);
        this.basePathCreated = false;
        this.basePath = (String)Preconditions.checkNotNull((Object)storagePath) + "/" + BLOB_PATH_NAME;
    }

    private void createBasePathIfNeeded() throws IOException {
        if (!this.basePathCreated) {
            LOG.info("Creating highly available BLOB storage directory at {}", (Object)this.basePath);
            this.fileSystem.mkdirs(new Path(this.basePath));
            LOG.debug("Created highly available BLOB storage directory at {}", (Object)this.basePath);
            this.basePathCreated = true;
        }
    }

    @Override
    public boolean put(File localFile, JobID jobId, BlobKey blobKey) throws IOException {
        this.createBasePathIfNeeded();
        String toBlobPath = BlobUtils.getStorageLocationPath(this.basePath, jobId, blobKey);
        try (FSDataOutputStream os = this.fileSystem.create(new Path(toBlobPath), FileSystem.WriteMode.OVERWRITE);){
            LOG.debug("Copying from {} to {}.", (Object)localFile, (Object)toBlobPath);
            Files.copy((File)localFile, (OutputStream)os);
            os.sync();
        }
        return true;
    }

    @Override
    public boolean get(JobID jobId, BlobKey blobKey, File localFile) throws IOException {
        return this.get(BlobUtils.getStorageLocationPath(this.basePath, jobId, blobKey), localFile, blobKey);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean get(String fromBlobPath, File toFile, BlobKey blobKey) throws IOException {
        Preconditions.checkNotNull((Object)fromBlobPath, (String)"Blob path");
        Preconditions.checkNotNull((Object)toFile, (String)"File");
        Preconditions.checkNotNull((Object)blobKey, (String)"Blob key");
        if (!toFile.exists() && !toFile.createNewFile()) {
            throw new IOException("Failed to create target file to copy to");
        }
        Path fromPath = new Path(fromBlobPath);
        MessageDigest md = BlobUtils.createMessageDigest();
        int buffSize = 4096;
        boolean success = false;
        try (FSDataInputStream is = this.fileSystem.open(fromPath);
             FileOutputStream fos = new FileOutputStream(toFile);){
            LOG.debug("Copying from {} to {}.", (Object)fromBlobPath, (Object)toFile);
            byte[] buf = new byte[4096];
            int bytesRead = is.read(buf);
            while (bytesRead >= 0) {
                fos.write(buf, 0, bytesRead);
                md.update(buf, 0, bytesRead);
                bytesRead = is.read(buf);
            }
            byte[] computedKey = md.digest();
            if (!Arrays.equals(computedKey, blobKey.getHash())) {
                throw new IOException("Detected data corruption during transfer");
            }
            success = true;
        }
        finally {
            if (!success) {
                try {
                    toFile.delete();
                }
                catch (Throwable throwable) {}
            }
        }
        return true;
    }

    @Override
    public boolean delete(JobID jobId, BlobKey blobKey) {
        return this.delete(BlobUtils.getStorageLocationPath(this.basePath, jobId, blobKey));
    }

    @Override
    public boolean deleteAll(JobID jobId) {
        return this.delete(BlobUtils.getStorageLocationPath(this.basePath, jobId));
    }

    private boolean delete(String blobPath) {
        try {
            LOG.debug("Deleting {}.", (Object)blobPath);
            Path path = new Path(blobPath);
            boolean result = true;
            if (this.fileSystem.exists(path)) {
                result = this.fileSystem.delete(path, true);
            } else {
                LOG.debug("The given path {} is not present anymore. No deletion is required.", (Object)path);
            }
            try {
                this.fileSystem.delete(path.getParent(), false);
                this.fileSystem.delete(new Path(this.basePath), false);
            }
            catch (IOException iOException) {
                // empty catch block
            }
            return result;
        }
        catch (Exception e) {
            LOG.warn("Failed to delete blob at " + blobPath);
            return false;
        }
    }

    @Override
    public void cleanupAllData() {
        try {
            LOG.debug("Cleaning up {}.", (Object)this.basePath);
            this.fileSystem.delete(new Path(this.basePath), true);
        }
        catch (Exception e) {
            LOG.error("Failed to clean up recovery directory.", (Throwable)e);
        }
    }

    @Override
    public void close() throws IOException {
    }
}

