// Licensed to the Apache Software Foundation (ASF) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The ASF licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

package org.apache.doris.nereids.trees.expressions.functions.scalar;

import org.apache.doris.catalog.FunctionSignature;
import org.apache.doris.nereids.trees.expressions.Expression;
import org.apache.doris.nereids.trees.expressions.functions.AlwaysNullable;
import org.apache.doris.nereids.trees.expressions.functions.CustomSignature;
import org.apache.doris.nereids.trees.expressions.functions.RewriteWhenAnalyze;
import org.apache.doris.nereids.trees.expressions.visitor.ExpressionVisitor;
import org.apache.doris.nereids.types.DataType;
import org.apache.doris.nereids.types.JsonType;
import org.apache.doris.nereids.types.VarcharType;
import org.apache.doris.nereids.util.ExpressionUtils;

import com.google.common.base.Preconditions;

import java.util.ArrayList;
import java.util.List;

/**
 * ScalarFunction 'json_replace'. This class is generated by GenerateFunction.
 */
public class JsonReplace extends ScalarFunction implements CustomSignature, AlwaysNullable, RewriteWhenAnalyze {

    /**
     * constructor with 3 or more arguments.
     */
    public JsonReplace(Expression arg0, Expression arg1, Expression arg2, Expression... varArgs) {
        super("json_replace", ExpressionUtils.mergeArguments(arg0, arg1, arg2, varArgs));
    }

    /** constructor for withChildren and reuse signature */
    private JsonReplace(ScalarFunctionParams functionParams) {
        super(functionParams);
    }

    @Override
    public FunctionSignature customSignature() {
        List<DataType> arguments = new ArrayList<>();
        arguments.add(JsonType.INSTANCE); // JSON object
        for (int i = 1; i < arity(); i++) {
            if ((i & 1) == 0) {
                arguments.add(getArgumentType(i));
            } else {
                arguments.add(VarcharType.SYSTEM_DEFAULT);
            }
        }
        return FunctionSignature.of(JsonType.INSTANCE, arguments);
    }

    /**
     * withChildren.
     */
    @Override
    public JsonReplace withChildren(List<Expression> children) {
        Preconditions.checkArgument(children.size() >= 3);
        return new JsonReplace(getFunctionParams(children));
    }

    @Override
    public <R, C> R accept(ExpressionVisitor<R, C> visitor, C context) {
        return visitor.visitJsonReplace(this, context);
    }

    @Override
    public Expression rewriteWhenAnalyze() {
        List<Expression> convectedChildren = new ArrayList<Expression>();
        convectedChildren.add(children.get(0));
        for (int i = 1; i < children.size(); i++) {
            Expression child = children.get(i);
            if (i % 2 == 1) {
                convectedChildren.add(child);
            } else if (child.getDataType() instanceof JsonType) {
                convectedChildren.add(child);
            } else {
                convectedChildren.add(new ToJson(child));
            }
        }
        return withChildren(convectedChildren);
    }
}
