/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.bcel.generic;

import java.io.DataInput;
import java.io.DataOutputStream;
import java.io.IOException;

import org.apache.bcel.classfile.AnnotationElementValue;
import org.apache.bcel.classfile.AnnotationEntry;
import org.apache.bcel.classfile.ArrayElementValue;
import org.apache.bcel.classfile.ClassElementValue;
import org.apache.bcel.classfile.ElementValue;
import org.apache.bcel.classfile.EnumElementValue;
import org.apache.bcel.classfile.SimpleElementValue;

/**
 * Generates element values in annotations.
 *
 * @since 6.0
 */
public abstract class ElementValueGen {

    /** Element value type: string. */
    public static final int STRING = 's';

    /** Element value type: enum constant. */
    public static final int ENUM_CONSTANT = 'e';

    /** Element value type: class. */
    public static final int CLASS = 'c';

    /** Element value type: annotation. */
    public static final int ANNOTATION = '@';

    /** Element value type: array. */
    public static final int ARRAY = '[';

    /** Element value type: primitive int. */
    public static final int PRIMITIVE_INT = 'I';

    /** Element value type: primitive byte. */
    public static final int PRIMITIVE_BYTE = 'B';

    /** Element value type: primitive char. */
    public static final int PRIMITIVE_CHAR = 'C';

    /** Element value type: primitive double. */
    public static final int PRIMITIVE_DOUBLE = 'D';

    /** Element value type: primitive float. */
    public static final int PRIMITIVE_FLOAT = 'F';

    /** Element value type: primitive long. */
    public static final int PRIMITIVE_LONG = 'J';

    /** Element value type: primitive short. */
    public static final int PRIMITIVE_SHORT = 'S';

    /** Element value type: primitive boolean. */
    public static final int PRIMITIVE_BOOLEAN = 'Z';

    /**
     * Creates an (modifiable) ElementValueGen copy of an (immutable) ElementValue - constant pool is assumed correct.
     *
     * @param value the element value to copy.
     * @param cpool the constant pool generator.
     * @param copyPoolEntries whether to copy pool entries.
     * @return a copy of the element value.
     */
    public static ElementValueGen copy(final ElementValue value, final ConstantPoolGen cpool, final boolean copyPoolEntries) {
        switch (value.getElementValueType()) {
        case 'B': // byte
        case 'C': // char
        case 'D': // double
        case 'F': // float
        case 'I': // int
        case 'J': // long
        case 'S': // short
        case 'Z': // boolean
        case 's': // String
            return new SimpleElementValueGen((SimpleElementValue) value, cpool, copyPoolEntries);
        case 'e': // Enum constant
            return new EnumElementValueGen((EnumElementValue) value, cpool, copyPoolEntries);
        case '@': // Annotation
            return new AnnotationElementValueGen((AnnotationElementValue) value, cpool, copyPoolEntries);
        case '[': // Array
            return new ArrayElementValueGen((ArrayElementValue) value, cpool, copyPoolEntries);
        case 'c': // Class
            return new ClassElementValueGen((ClassElementValue) value, cpool, copyPoolEntries);
        default:
            throw new UnsupportedOperationException("Not implemented yet! (" + value.getElementValueType() + ")");
        }
    }

    /**
     * Reads an element value from a DataInput.
     *
     * @param dis the data input stream.
     * @param cpGen the constant pool.
     * @return the element value read.
     * @throws IOException if an I/O error occurs.
     */
    public static ElementValueGen readElementValue(final DataInput dis, final ConstantPoolGen cpGen) throws IOException {
        final int type = dis.readUnsignedByte();
        switch (type) {
        case 'B': // byte
            return new SimpleElementValueGen(PRIMITIVE_BYTE, dis.readUnsignedShort(), cpGen);
        case 'C': // char
            return new SimpleElementValueGen(PRIMITIVE_CHAR, dis.readUnsignedShort(), cpGen);
        case 'D': // double
            return new SimpleElementValueGen(PRIMITIVE_DOUBLE, dis.readUnsignedShort(), cpGen);
        case 'F': // float
            return new SimpleElementValueGen(PRIMITIVE_FLOAT, dis.readUnsignedShort(), cpGen);
        case 'I': // int
            return new SimpleElementValueGen(PRIMITIVE_INT, dis.readUnsignedShort(), cpGen);
        case 'J': // long
            return new SimpleElementValueGen(PRIMITIVE_LONG, dis.readUnsignedShort(), cpGen);
        case 'S': // short
            return new SimpleElementValueGen(PRIMITIVE_SHORT, dis.readUnsignedShort(), cpGen);
        case 'Z': // boolean
            return new SimpleElementValueGen(PRIMITIVE_BOOLEAN, dis.readUnsignedShort(), cpGen);
        case 's': // String
            return new SimpleElementValueGen(STRING, dis.readUnsignedShort(), cpGen);
        case 'e': // Enum constant
            return new EnumElementValueGen(dis.readUnsignedShort(), dis.readUnsignedShort(), cpGen);
        case 'c': // Class
            return new ClassElementValueGen(dis.readUnsignedShort(), cpGen);
        case '@': // Annotation
            // TODO: isRuntimeVisible ??????????
            // FIXME
            return new AnnotationElementValueGen(ANNOTATION, new AnnotationEntryGen(AnnotationEntry.read(dis, cpGen.getConstantPool(), true), cpGen, false),
                cpGen);
        case '[': // Array
            final int numArrayVals = dis.readUnsignedShort();
            final ElementValue[] evalues = new ElementValue[numArrayVals];
            for (int j = 0; j < numArrayVals; j++) {
                evalues[j] = ElementValue.readElementValue(dis, cpGen.getConstantPool());
            }
            return new ArrayElementValueGen(ARRAY, evalues, cpGen);
        default:
            throw new IllegalArgumentException("Unexpected element value kind in annotation: " + type);
        }
    }

    /**
     * @deprecated (since 6.0) will be made private and final; do not access directly, use getter
     */
    @Deprecated
    protected int type;

    /**
     * @deprecated (since 6.0) will be made private and final; do not access directly, use getter
     */
    @Deprecated
    protected ConstantPoolGen cpGen;

    /**
     * Constructs an ElementValueGen.
     *
     * @param type the element value type.
     * @param cpGen the constant pool.
     */
    protected ElementValueGen(final int type, final ConstantPoolGen cpGen) {
        this.type = type;
        this.cpGen = cpGen;
    }

    /**
     * Dumps this element value to a DataOutputStream.
     *
     * @param dos the output stream.
     * @throws IOException if an I/O error occurs.
     */
    public abstract void dump(DataOutputStream dos) throws IOException;

    /**
     * Gets the constant pool.
     *
     * @return the constant pool.
     */
    protected ConstantPoolGen getConstantPool() {
        return cpGen;
    }

    /**
     * Subtypes return an immutable variant of the ElementValueGen.
     *
     * @return an immutable variant of the ElementValueGen.
     */
    public abstract ElementValue getElementValue();

    /**
     * Gets the element value type.
     *
     * @return the element value type.
     */
    public int getElementValueType() {
        return type;
    }

    /**
     * Returns a string representation of the element value.
     *
     * @return a string representation of the element value.
     */
    public abstract String stringifyValue();
}
